import React from 'react';
import { DocumentImage } from './components/DocumentImage';
import { TypeSelector } from './components/TypeSelector';
import { ProgressBar } from './components/ProgressBar';
import { fetchNextDocument, updateDocumentType } from './services/api';
import { Document } from './types';
import { FileCheck } from 'lucide-react';

function App() {
  const [document, setDocument] = React.useState<Document | null>(null);
  const [selectedType, setSelectedType] = React.useState<string>('');
  const [isLoading, setIsLoading] = React.useState(false);
  const [error, setError] = React.useState<string | null>(null);
  const [progress, setProgress] = React.useState({ current: 0, total: 100 });

  const loadNextDocument = React.useCallback(async () => {
    setIsLoading(true);
    setError(null);
    
    const response = await fetchNextDocument();
    
    if (response.error) {
      setError(response.error);
    } else if (response.data) {
      setDocument(response.data);
      setSelectedType('');
    } else {
      setDocument(null);
    }
    
    setIsLoading(false);
  }, []);

  React.useEffect(() => {
    loadNextDocument();
  }, [loadNextDocument]);

  const handleSubmit = async () => {
    if (!document || !selectedType) return;

    setIsLoading(true);
    const response = await updateDocumentType(document.document_id, selectedType);
    
    if (response.error) {
      setError(response.error);
      setIsLoading(false);
      return;
    }

    setProgress(prev => ({
      ...prev,
      current: Math.min(prev.current + 1, prev.total)
    }));
    
    await loadNextDocument();
  };

  if (!document && !error && !isLoading) {
    return (
      <div className="min-h-screen bg-gray-100 flex items-center justify-center p-4">
        <div className="bg-white rounded-lg shadow-lg p-8 max-w-md w-full text-center">
          <FileCheck className="w-16 h-16 text-green-500 mx-auto mb-4" />
          <h2 className="text-2xl font-bold text-gray-900 mb-2">All Done!</h2>
          <p className="text-gray-600">All documents have been categorized.</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-100 p-4 sm:p-6 lg:p-8">
      <div className="max-w-4xl mx-auto">
        <div className="bg-white rounded-lg shadow-lg p-6 mb-6">
          <h1 className="text-2xl font-bold text-gray-900 mb-6">
            Document Categorization
          </h1>
          
          <ProgressBar
            current={progress.current}
            total={progress.total}
          />
        </div>

        {error ? (
          <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
            <p className="text-red-800">{error}</p>
            <button
              onClick={loadNextDocument}
              className="mt-2 text-red-600 hover:text-red-800 font-medium"
            >
              Try Again
            </button>
          </div>
        ) : null}

        <div className="bg-white rounded-lg shadow-lg p-6 mb-6">
          {document && (
            <DocumentImage
              src={document.filepath}
              onRetry={loadNextDocument}
            />
          )}
        </div>

        <div className="bg-white rounded-lg shadow-lg p-6 mb-6">
          <h2 className="text-lg font-semibold text-gray-900 mb-4">
            Select Document Type
          </h2>
          
          <TypeSelector
            selectedType={selectedType}
            onChange={setSelectedType}
          />
        </div>

        <button
          onClick={handleSubmit}
          disabled={!selectedType || isLoading}
          className="w-full bg-blue-500 text-white py-3 px-6 rounded-lg font-medium
                   hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
                   disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
        >
          {isLoading ? 'Processing...' : 'Submit and Continue'}
        </button>
      </div>
    </div>
  );
}

export default App;